/* ***************************************************************************+
 * ITX package (cnrg.itx) for telephony application programming.              *
 * Copyright (c) 1999  Cornell University, Ithaca NY.                         *
 * A copy of the license is distributed with this package.  Look in the docs  *
 * directory, filename GPL.  Contact information: bergmark@cs.cornell.edu     *
 ******************************************************************************/
#include "stdafx.h"
#include <stdlib.h>
#include <malloc.h>
#include <errno.h>
#include <sys/types.h>
#include <arpa/inet.h>
#include <arpa/nameser.h>
#include <resolv.h>
#include "netdb.h"
#include "cnrg_itx_ds_DSComm.h"

/* Options.  Leave them on. */
#define DEBUG

#define ShrinkBuffer(x)  if ((buflen -= x) < 0) return (-2);

// function definitions
static int getnum_str(u_char **, u_char *);
static int getword_str(char *, int, u_char **, u_char *);

// global variable
char *m_pfilepath;
int m_bSetPath;

/*
 * Form update packets.
 * Returns the size of the resulting packet if no error
 * On error,
 *	returns -1 if error in reading a word/number in rdata
 *		   portion for update packets
 *		-2 if length of buffer passed is insufficient
 *		-3 if zone section is not the first section in
 *		   the linked list, or section order has a problem
 *		-4 on a number overflow
 *		-5 unknown operation or no records
 */
int
res_mkupdate(ns_updrec *rrecp_in, u_char *buf, int buflen) {
	ns_updrec *rrecp_start = rrecp_in;
	HEADER *hp;
	u_char *cp, *sp1, *sp2, *startp, *endp;
	int n, i, soanum, multiline;
	ns_updrec *rrecp, *recptr = NULL;
	struct in_addr ina;
        char buf2[MAXDNAME];
	int section, numrrs = 0, counts[ns_s_max];
	u_int16_t rtype, rclass;
	u_int32_t n1, rttl;
	u_char *dnptrs[20], **dpp, **lastdnptr;

	if ((_res.options & RES_INIT) == 0 && res_init() == -1) {
// Added #define
#ifndef WINNT
		h_errno = NETDB_INTERNAL;
#else
		WSASetLastError(NETDB_INTERNAL);
#endif
		return (-1);
	}

	/*
	 * Initialize header fields.
	 */
	if ((buf == NULL) || (buflen < HFIXEDSZ))
		return (-1);
	memset(buf, 0, HFIXEDSZ);
	hp = (HEADER *) buf;
	hp->id = htons(++_res.id);
	hp->opcode = ns_o_update;
	hp->rcode = NOERROR;
	sp1 = buf + 2*INT16SZ;  /* save pointer to zocount */
	cp = buf + HFIXEDSZ;
	buflen -= HFIXEDSZ;
	dpp = dnptrs;
	*dpp++ = buf;
	*dpp++ = NULL;
	lastdnptr = dnptrs + sizeof dnptrs / sizeof dnptrs[0];

	if (rrecp_start == NULL)
		return (-5);
	else if (rrecp_start->r_section != S_ZONE)
		return (-3);

	memset(counts, 0, sizeof counts);
	for (rrecp = rrecp_start; rrecp; rrecp = rrecp->r_grpnext) {
		numrrs++;
                section = rrecp->r_section;
		if (section < 0 || section >= ns_s_max)
			return (-1);
		counts[section]++;
		for (i = section + 1; i < ns_s_max; i++)
			if (counts[i])
				return (-3);
		rtype = rrecp->r_type;
		rclass = rrecp->r_class;
		rttl = rrecp->r_ttl;
		/* overload class and type */
		if (section == S_PREREQ) {
			rttl = 0;
			switch (rrecp->r_opcode) {
			case YXDOMAIN:
				rclass = C_ANY;
				rtype = T_ANY;
				rrecp->r_size = 0;
				break;
			case NXDOMAIN:
				rclass = C_NONE;
				rtype = T_ANY;
				rrecp->r_size = 0;
				break;
			case NXRRSET:
				rclass = C_NONE;
				rrecp->r_size = 0;
				break;
			case YXRRSET:
				if (rrecp->r_size == 0)
					rclass = C_ANY;
				break;
			default:
				fprintf(stderr,
					"res_mkupdate: incorrect opcode: %d\n",
					rrecp->r_opcode);
				fflush(stderr);
				return (-1);
			}
		} else if (section == ns_s_ud) {
			switch (rrecp->r_opcode) {
			case ns_uop_delete:
				rclass = rrecp->r_size == 0 ? C_ANY : C_NONE;
				break;
			case ns_uop_add:
				break;
			default:
				fprintf(stderr,
					"res_mkupdate: incorrect opcode: %d\n",
					rrecp->r_opcode);
				fflush(stderr);
				return (-1);
			}
		}

		/*
		 * XXX	appending default domain to owner name is omitted,
		 *	fqdn must be provided
		 */
		if ((n = dn_comp(rrecp->r_dname, cp, buflen, dnptrs,
				 lastdnptr)) < 0)
			return (-1);
		cp += n;
		ShrinkBuffer(n + 2*INT16SZ);
		PUTSHORT(rtype, cp);
		PUTSHORT(rclass, cp);
		if (section == S_ZONE) {
			if (numrrs != 1 || rrecp->r_type != T_SOA)
				return (-3);
			continue;
		}
		ShrinkBuffer(INT32SZ + INT16SZ);
		PUTLONG(rttl, cp);
		sp2 = cp;  /* save pointer to length byte */
		cp += INT16SZ;
		if (rrecp->r_size == 0) {
			if (section == ns_s_ud && rclass != C_ANY)
				return (-1);
			else {
				PUTSHORT(0, sp2);
				continue;
			}
		}
		startp = rrecp->r_data;
		endp = startp + rrecp->r_size - 1;
		/* XXX this should be done centrally. */
		switch (rrecp->r_type) {
		case T_A:
			if (!getword_str(buf2, sizeof buf2, &startp, endp))
				return (-1);
			if (!inet_aton(buf2, &ina))
				return (-1);
			n1 = ntohl(ina.s_addr);
			ShrinkBuffer(INT32SZ);
			PUTLONG(n1, cp);
			break;
		case T_CNAME:
		case T_MB:
		case T_MG:
		case T_MR:
		case T_NS:
		case T_PTR:
			if (!getword_str(buf2, sizeof buf2, &startp, endp))
				return (-1);
			n = dn_comp(buf2, cp, buflen, dnptrs, lastdnptr);
			if (n < 0)
				return (-1);
			cp += n;
			ShrinkBuffer(n);
			break;
		case T_MINFO:
		case T_SOA:
		case T_RP:
			for (i = 0; i < 2; i++) {
				if (!getword_str(buf2, sizeof buf2, &startp,
						 endp))
				return (-1);
				n = dn_comp(buf2, cp, buflen,
					    dnptrs, lastdnptr);
				if (n < 0)
					return (-1);
				cp += n;
				ShrinkBuffer(n);
			}
			if (rrecp->r_type == T_SOA) {
				ShrinkBuffer(5 * INT32SZ);
				while (isspace(*startp) || !*startp)
					startp++;
				if (*startp == '(') {
					multiline = 1;
					startp++;
				} else
					multiline = 0;
				/* serial, refresh, retry, expire, minimum */
				for (i = 0; i < 5; i++) {
					soanum = getnum_str(&startp, endp);
					if (soanum < 0)
						return (-1);
					PUTLONG(soanum, cp);
				}
				if (multiline) {
					while (isspace(*startp) || !*startp)
						startp++;
					if (*startp != ')')
						return (-1);
				}
			}
			break;
		case T_MX:
		case T_AFSDB:
		case T_RT:
			n = getnum_str(&startp, endp);
			if (n < 0)
				return (-1);
			PUTSHORT(n, cp);
			ShrinkBuffer(INT16SZ);
			if (!getword_str(buf2, sizeof buf2, &startp, endp))
				return (-1);
			n = dn_comp(buf2, cp, buflen, dnptrs, lastdnptr);
			if (n < 0)
				return (-1);
			cp += n;
			ShrinkBuffer(n);
			break;
		case T_PX:
			n = getnum_str(&startp, endp);
			if (n < 0)
				return (-1);
			PUTSHORT(n, cp);
			ShrinkBuffer(INT16SZ);
			for (i = 0; i < 2; i++) {
				if (!getword_str(buf2, sizeof buf2, &startp,
						 endp))
					return (-1);
				n = dn_comp(buf2, cp, buflen, dnptrs,
					    lastdnptr);
				if (n < 0)
					return (-1);
				cp += n;
				ShrinkBuffer(n);
			}
			break;
		case T_WKS:
		case T_HINFO:
		case T_TXT:
		case T_X25:
		case T_ISDN:
		case T_NSAP:
		case T_LOC:
			/* XXX - more fine tuning needed here */
			ShrinkBuffer(rrecp->r_size);
			memcpy(cp, rrecp->r_data, rrecp->r_size);
			cp += rrecp->r_size;
			break;
		default:
			return (-1);
		} /*switch*/
		n = (u_int16_t)((cp - sp2) - INT16SZ);
		PUTSHORT(n, sp2);
	} /*for*/
		
	hp->qdcount = htons(counts[0]);
	hp->ancount = htons(counts[1]);
	hp->nscount = htons(counts[2]);
	hp->arcount = htons(counts[3]);
	return (cp - buf);
}


/*
 * Get a whitespace delimited word from a string (not file)
 * into buf. modify the start pointer to point after the
 * word in the string.
 */
static int
getword_str(char *buf, int size, u_char **startpp, u_char *endp) {
        char *cp;
        int c;
 
        for (cp = buf; *startpp <= endp; ) {
                c = **startpp;
                if (isspace(c) || c == '\0') {
                        if (cp != buf) /* trailing whitespace */
                                break;
                        else { /* leading whitespace */
                                (*startpp)++;
                                continue;
                        }
                }
                (*startpp)++;
                if (cp >= buf+size-1)
                        break;
                *cp++ = (u_char)c;
        }
        *cp = '\0';
        return (cp != buf);
}


/*
 * Get a whitespace delimited number from a string (not file) into buf
 * update the start pointer to point after the number in the string.
 */
static int
getnum_str(u_char **startpp, u_char *endp) {
        int c, n;
        int seendigit = 0;
        int seendecimal = 0;
        int m = 0;

        for (n = 0; *startpp <= endp; ) {
                c = **startpp;
                if (isspace(c) || c == '\0') {
                        if (seendigit) /* trailing whitespace */
                                break;
                        else { /* leading whitespace */
                                (*startpp)++;
                                continue;
                        }
                }
                if (c == ';') {
                        while ((*startpp <= endp) &&
			       ((c = **startpp) != '\n'))
					(*startpp)++;
                        if (seendigit)
                                break;
                        continue;
                }
                if (!isdigit(c)) {
                        if (c == ')' && seendigit) {
                                (*startpp)--;
                                break;
                        }
			return (-1);
                }        
                (*startpp)++;
                n = n * 10 + (c - '0');
                seendigit = 1;
        }
        return (n + m);
}


/*
 * Allocate a resource record buffer & save rr info.
 */
ns_updrec *
res_mkupdrec(int section, const char *dname,
	     u_int nclass, u_int type, u_long ttl) {	// changed class to nclass
	ns_updrec *rrecp = (ns_updrec *)calloc(1, sizeof(ns_updrec));

	if (!rrecp || !(rrecp->r_dname = strdup(dname)))
		return (NULL);
 	rrecp->r_class = nclass;
	rrecp->r_type = type;
	rrecp->r_ttl = ttl;
	rrecp->r_section = section;
	return (rrecp);
}


/*
 * Free a resource record buffer created by res_mkupdrec.
 */
void
res_freeupdrec(ns_updrec *rrecp) {
	/* Note: freeing r_dp is the caller's responsibility. */
	if (rrecp->r_dname != NULL)
		free(rrecp->r_dname);
	free(rrecp);
}


/*
 * Separate a linked list of records into groups so that all records
 * in a group will belong to a single zone on the nameserver.
 * Create a dynamic update packet for each zone and send it to the
 * nameservers for that zone, and await answer.
 * Abort if error occurs in updating any zone.
 * Return the number of zones updated on success, < 0 on error.
 *
 * On error, caller must deal with the unsynchronized zones
 * eg. an A record might have been successfully added to the forward
 * zone but the corresponding PTR record would be missing if error
 * was encountered while updating the reverse zone.
 */
#define NSMAX 16

struct ns1 {
	char nsname[MAXDNAME];
	struct in_addr nsaddr1;
};

struct zonegrp {
	char 		z_origin[MAXDNAME];
	int16_t		z_class;
	char		z_soardata[MAXDNAME + 5 * INT32SZ];
	struct ns1 	z_ns[NSMAX];
	int		z_nscount;
	ns_updrec *	z_rr;
	struct zonegrp *z_next;
};


int
res_update(ns_updrec *rrecp_in) {
	ns_updrec *rrecp, *tmprrecp;
	u_char buf[PACKETSZ], answer[PACKETSZ], packet[2*PACKETSZ];
	char name[MAXDNAME], zname[MAXDNAME], primary[MAXDNAME],
	     mailaddr[MAXDNAME];
	u_char soardata[2*MAXCDNAME+5*INT32SZ];
	char *dname, *svdname, *cp1, *target;
	u_char *cp, *eom;
	HEADER *hp = (HEADER *) answer;
	struct zonegrp *zptr = NULL, *tmpzptr, *prevzptr, *zgrp_start = NULL;
	int i, j, k = 0, n, ancount, nscount, arcount, rcode, rdatasize,
	    newgroup, done, myzone, seen_before, numzones = 0;
	u_int16_t dlen, nclass, qclass, type, qtype;	// change class to nclass
	u_int32_t ttl;

	if ((_res.options & RES_INIT) == 0 && res_init() == -1) {
// Added #define
#ifndef WINNT
		h_errno = NETDB_INTERNAL;
#else
		WSASetLastError(NETDB_INTERNAL);
#endif
		return (-1);
	}

	for (rrecp = rrecp_in; rrecp; rrecp = rrecp->r_next) {
		dname = rrecp->r_dname;
		n = strlen(dname);
		if (dname[n-1] == '.')
			dname[n-1] = '\0';
		qtype = T_SOA;
		qclass = rrecp->r_class;
		done = 0;
		seen_before = 0;

		while (!done && dname) {
		    if (qtype == T_SOA) {
			for (tmpzptr = zgrp_start;
			     tmpzptr && !seen_before;
			     tmpzptr = tmpzptr->z_next) {
				if (strcasecmp(dname,
					       tmpzptr->z_origin) == 0 &&
				    tmpzptr->z_class == qclass)
					seen_before++;
				for (tmprrecp = tmpzptr->z_rr;
				     tmprrecp && !seen_before;
				     tmprrecp = tmprrecp->r_grpnext)
				if (strcasecmp(dname, tmprrecp->r_dname) == 0
				    && tmprrecp->r_class == qclass) {
					seen_before++;
					break;
				}
				if (seen_before) {
					/*
					 * Append to the end of
					 * current group.
					 */
					for (tmprrecp = tmpzptr->z_rr;
					     tmprrecp->r_grpnext;
					     tmprrecp = tmprrecp->r_grpnext)
						(void)NULL;
					tmprrecp->r_grpnext = rrecp;
					rrecp->r_grpnext = NULL;
					done = 1;
					break;
				}
			}
		} else if (qtype == T_A) {
		    for (tmpzptr = zgrp_start;
			 tmpzptr && !done;
			 tmpzptr = tmpzptr->z_next)
			    for (i = 0; i < tmpzptr->z_nscount; i++)
				if (tmpzptr->z_class == qclass &&
				    strcasecmp(tmpzptr->z_ns[i].nsname,
					       dname) == 0 &&
				    tmpzptr->z_ns[i].nsaddr1.s_addr != 0) {
					zptr->z_ns[k].nsaddr1.s_addr =
					 tmpzptr->z_ns[i].nsaddr1.s_addr;
					done = 1;
					break;
				}
		}
		if (done)
		    break;
		n = res_mkquery(QUERY, dname, qclass, qtype, NULL,
				0, NULL, buf, sizeof buf);
		if (n <= 0) {
		    fprintf(stderr, "res_update: mkquery failed\n");
		    return (n);
		}
		n = res_send(buf, n, answer, sizeof answer);
		if (n < 0) {
		    fprintf(stderr, "res_update: send error for %s\n",
			    rrecp->r_dname);
		    return (n);
		}
		if (n < HFIXEDSZ)
			return (-1);
		ancount = ntohs(hp->ancount);
		nscount = ntohs(hp->nscount);
		arcount = ntohs(hp->arcount);
		rcode = hp->rcode;
		cp = answer + HFIXEDSZ;
		eom = answer + n;
		/* skip the question section */
		n = dn_skipname(cp, eom);
		if (n < 0 || cp + n + 2 * INT16SZ > eom)
			return (-1);
		cp += n + 2 * INT16SZ;

		if (qtype == T_SOA) {
		    if (ancount == 0 && nscount == 0 && arcount == 0) {
			/*
			 * if (rcode == NOERROR) then the dname exists but
			 * has no soa record associated with it.
			 * if (rcode == NXDOMAIN) then the dname does not
			 * exist and the server is replying out of NCACHE.
			 * in either case, proceed with the next try
			 */
			dname = strchr(dname, '.');
			if (dname != NULL)
				dname++;
			continue;
		    } else if ((rcode == NOERROR || rcode == NXDOMAIN) &&
			       ancount == 0 &&
			       nscount == 1 && arcount == 0) {
			/*
			 * name/data does not exist, soa record supplied in the
			 * authority section
			 */
			/* authority section must contain the soa record */
			if ((n = dn_expand(answer, eom, cp, zname,
					sizeof zname)) < 0)
			    return (n);
			cp += n;
			if (cp + 2 * INT16SZ > eom)
				return (-1);
			GETSHORT(type, cp);
			GETSHORT(nclass, cp);
			if (type != T_SOA || nclass != qclass) {
			    fprintf(stderr, "unknown answer\n");
			    return (-1);
			}
			myzone = 0;
			svdname = dname;
			while (dname)
			    if (strcasecmp(dname, zname) == 0) {
				myzone = 1;
				break;
			    } else if ((dname = strchr(dname, '.')) != NULL)
				dname++;
			if (!myzone) {
			    dname = strchr(svdname, '.');
			    if (dname != NULL)
				dname++;
			    continue;
			}
			nscount = 0;
			/* fallthrough */
		    } else if (rcode == NOERROR && ancount == 1) {
			/*
			 * found the zone name
			 * new servers will supply NS records for the zone
			 * in authority section and A records for those 
			 * nameservers in the additional section
			 * older servers have to be explicitly queried for
			 * NS records for the zone
			 */
			/* answer section must contain the soa record */
			if ((n = dn_expand(answer, eom, cp, zname,
			 	       	   sizeof zname)) < 0)
				return (n);
			else
				cp += n;
			if (cp + 2 * INT16SZ > eom)
				return (-1);
			GETSHORT(type, cp);
			GETSHORT(nclass, cp);
			if (type == T_CNAME) {
				dname = strchr(dname, '.');
				if (dname != NULL)
					dname++;
				continue;
			}
			if (strcasecmp(dname, zname) != 0 ||
			    type != T_SOA ||
			    nclass != rrecp->r_class) {
				fprintf(stderr, "unknown answer\n");
				return (-1);
			}
			/* FALLTHROUGH */
		    } else {
			fprintf(stderr,
		"unknown response: ans=%d, auth=%d, add=%d, rcode=%d\n",
				ancount, nscount, arcount, hp->rcode);
			return (-1);
		    }
		    if (cp + INT32SZ + INT16SZ > eom)
			    return (-1);
		    /* continue processing the soa record */
		    GETLONG(ttl, cp);
		    GETSHORT(dlen, cp);
		    if (cp + dlen > eom)
			    return (-1);
		    newgroup = 1;
		    zptr = zgrp_start;
		    prevzptr = NULL;
		    while (zptr) {
			if (strcasecmp(zname, zptr->z_origin) == 0 &&
			    type == T_SOA && nclass == qclass) {
				newgroup = 0;
				break;
			}
			prevzptr = zptr;
			zptr = zptr->z_next;
		    }
		    if (!newgroup) {
			for (tmprrecp = zptr->z_rr;
			     tmprrecp->r_grpnext;
			     tmprrecp = tmprrecp->r_grpnext)
				    ;
			tmprrecp->r_grpnext = rrecp;
			rrecp->r_grpnext = NULL;
			done = 1;
			cp += dlen;
			break;
		    } else {
			if ((n = dn_expand(answer, eom, cp, primary,
				       	   sizeof primary)) < 0)
			    return (n);
			cp += n;
			/* 
			 * We don't have to bounds check here because the
			 * next use of 'cp' is in dn_expand().
			 */
			cp1 = (char *)soardata;
			strcpy(cp1, primary);
			cp1 += strlen(cp1) + 1;
			if ((n = dn_expand(answer, eom, cp, mailaddr,
				       	   sizeof mailaddr)) < 0)
			    return (n);
			cp += n;
			strcpy(cp1, mailaddr);
			cp1 += strlen(cp1) + 1;
			if (cp + 5*INT32SZ > eom)
				return (-1);
			memcpy(cp1, cp, 5*INT32SZ);
			cp += 5*INT32SZ;
			cp1 += 5*INT32SZ;
			rdatasize = (u_char *)cp1 - soardata;
			zptr = (struct zonegrp *)calloc(1, sizeof(struct zonegrp));
			if (zptr == NULL)
                	    return (-1);
			if (zgrp_start == NULL)
			    zgrp_start = zptr;
			else
			    prevzptr->z_next = zptr;
			zptr->z_rr = rrecp;
			rrecp->r_grpnext = NULL;
			strcpy(zptr->z_origin, zname);
			zptr->z_class = nclass;
			memcpy(zptr->z_soardata, soardata, rdatasize);
			/* fallthrough to process NS and A records */
		    }
		} else if (qtype == T_NS) {
		    if (rcode == NOERROR && ancount > 0) {
			strcpy(zname, dname);
			for (zptr = zgrp_start; zptr; zptr = zptr->z_next) {
			    if (strcasecmp(zname, zptr->z_origin) == 0)
				break;
			}
			if (zptr == NULL)
			    /* should not happen */
			    return (-1);
			if (nscount > 0) {
			    /*
			     * answer and authority sections contain
			     * the same information, skip answer section
			     */
			    for (j = 0; j < ancount; j++) {
				n = dn_skipname(cp, eom);
				if (n < 0)
					return (-1);
				n += 2*INT16SZ + INT32SZ;
				if (cp + n + INT16SZ > eom)
					return (-1);
				cp += n;
				GETSHORT(dlen, cp);
				cp += dlen;
			    }
			} else
			    nscount = ancount;
			/* fallthrough to process NS and A records */
		    } else {
			fprintf(stderr, "cannot determine nameservers for %s:\
ans=%d, auth=%d, add=%d, rcode=%d\n",
				dname, ancount, nscount, arcount, hp->rcode);
			return (-1);
		    }
		} else if (qtype == T_A) {
		    if (rcode == NOERROR && ancount > 0) {
			arcount = ancount;
			ancount = nscount = 0;
			/* fallthrough to process A records */
		    } else {
			fprintf(stderr, "cannot determine address for %s:\
ans=%d, auth=%d, add=%d, rcode=%d\n",
				dname, ancount, nscount, arcount, hp->rcode);
			return (-1);
		    }
		}
		/* process NS records for the zone */
		j = 0;
		for (i = 0; i < nscount; i++) {
		    if ((n = dn_expand(answer, eom, cp, name,
					sizeof name)) < 0)
			return (n);
		    cp += n;
		    if (cp + 3 * INT16SZ + INT32SZ > eom)
			    return (-1);
		    GETSHORT(type, cp);
		    GETSHORT(nclass, cp);
		    GETLONG(ttl, cp);
		    GETSHORT(dlen, cp);
		    if (cp + dlen > eom)
			return (-1);
		    if (strcasecmp(name, zname) == 0 &&
			type == T_NS && nclass == qclass) {
				if ((n = dn_expand(answer, eom, cp,
						   name, sizeof name)) < 0)
					return (n);
			    target = zptr->z_ns[j++].nsname;
			    strcpy(target, name);
		    }
		    cp += dlen;
		}
		if (zptr->z_nscount == 0)
		    zptr->z_nscount = j;
		/* get addresses for the nameservers */
		for (i = 0; i < arcount; i++) {
		    if ((n = dn_expand(answer, eom, cp, name,
					sizeof name)) < 0)
			return (n);
		    cp += n;
		    if (cp + 3 * INT16SZ + INT32SZ > eom)
			return (-1);
		    GETSHORT(type, cp);
		    GETSHORT(nclass, cp);
		    GETLONG(ttl, cp);
		    GETSHORT(dlen, cp);
		    if (cp + dlen > eom)
			    return (-1);
		    if (type == T_A && dlen == INT32SZ && nclass == qclass) {
			for (j = 0; j < zptr->z_nscount; j++)
			    if (strcasecmp(name, zptr->z_ns[j].nsname) == 0) {
				memcpy(&zptr->z_ns[j].nsaddr1.s_addr, cp,
				       INT32SZ);
				break;
			    }
		    }
		    cp += dlen;
		}
		if (zptr->z_nscount == 0) {
		    dname = zname;
		    qtype = T_NS;
		    continue;
		}
		done = 1;
		for (k = 0; k < zptr->z_nscount; k++)
		    if (zptr->z_ns[k].nsaddr1.s_addr == 0) {
			done = 0;
			dname = zptr->z_ns[k].nsname;
			qtype = T_A;
		    }

 	    } /* while */
	}

//	_res.options |= RES_DEBUG;
	for (zptr = zgrp_start; zptr; zptr = zptr->z_next) {

		/* append zone section */
		rrecp = res_mkupdrec(ns_s_zn, zptr->z_origin,
				     zptr->z_class, ns_t_soa, 0);
		if (rrecp == NULL) {
			fprintf(stderr, "saverrec error\n");
			fflush(stderr);
			return (-1);
		}
		rrecp->r_grpnext = zptr->z_rr;
		zptr->z_rr = rrecp;

		n = res_mkupdate(zptr->z_rr, packet, sizeof packet);
		if (n < 0) {
			fprintf(stderr, "res_mkupdate error\n");
			fflush(stderr);
			return (-1);
		} else
		  ;
//			fprintf(stdout, "res_mkupdate: packet size = %d\n", n);

		/*
		 * Override the list of NS records from res_init() with
		 * the authoritative nameservers for the zone being updated.
		 * Sort primary to be the first in the list of nameservers.
		 */
		for (i = 0; i < zptr->z_nscount; i++) {
			if (strcasecmp(zptr->z_ns[i].nsname,
				       zptr->z_soardata) == 0) {
				struct in_addr tmpaddr;

				if (i != 0) {
					strcpy(zptr->z_ns[i].nsname,
					       zptr->z_ns[0].nsname);
					strcpy(zptr->z_ns[0].nsname,
					       zptr->z_soardata);
					tmpaddr = zptr->z_ns[i].nsaddr1;
					zptr->z_ns[i].nsaddr1 =
						zptr->z_ns[0].nsaddr1;
					zptr->z_ns[0].nsaddr1 = tmpaddr;
				}
				break;
			}
		}
		for (i = 0; i < MAXNS; i++) {
			_res.nsaddr_list[i].sin_addr = zptr->z_ns[i].nsaddr1;
			_res.nsaddr_list[i].sin_family = AF_INET;
			_res.nsaddr_list[i].sin_port = htons(NAMESERVER_PORT);
		}
		_res.nscount = (zptr->z_nscount < MAXNS) ? 
					zptr->z_nscount : MAXNS;
		n = res_send(packet, n, answer, sizeof(answer));
		if (n < 0) {
			fprintf(stderr, "res_send: send error, n=%d\n", n);
			break;
		} else
			numzones++;
	}

	/* free malloc'ed memory */
	while(zgrp_start) {
		zptr = zgrp_start;
		zgrp_start = zgrp_start->z_next;
		res_freeupdrec(zptr->z_rr);  /* Zone section we allocated. */
		free((char *)zptr);
	}

	return (numzones);
}


static int
skiprr(const u_char *ptr, const u_char *eom, ns_sect section, int count) {
	const u_char *optr = ptr;

	for ((void)NULL; count > 0; count--) {
		int b, rdlength;

		b = dn_skipname(ptr, eom);
		if (b < 0)
			goto emsgsize;
		ptr += b/*Name*/ + INT16SZ/*Type*/ + INT16SZ/*Class*/;
		if (section != ns_s_qd) {
			if (ptr + INT32SZ > eom)
				goto emsgsize;
			ptr += INT32SZ/*TTL*/;
			if (ptr + INT16SZ > eom)
				goto emsgsize;
			GETSHORT(rdlength, ptr);	// change NS_GET16 to GETSHORT
			ptr += rdlength/*RData*/;
		}
	}
	if (ptr > eom)
		goto emsgsize;
	return (ptr - optr);
 emsgsize:
// Added #ifndef
#ifndef WINNT
     errno = EMSGSIZE;
#else
     WSASetLastError(WSAEMSGSIZE);
     SetLastError(WSAEMSGSIZE);
#endif
	return (-1);
}


int
ns_parserr(ns_msg *handle, ns_sect section, int rrnum, ns_rr *rr) {
	int b;

	/* Make section right. */
	if (section < 0 || section >= ns_s_max)
		goto enodev;
	if ((int)section != (int)handle->_sect) {
		handle->_sect = section;
		handle->_rrnum = 0;
		handle->_ptr = handle->_sections[(int)section];
	}

	/* Make rrnum right. */
	if (rrnum == -1)
		rrnum = handle->_rrnum;
	if (rrnum < 0 || rrnum >= handle->_counts[(int)section])
		goto enodev;
	if (rrnum < handle->_rrnum) {
		handle->_rrnum = 0;
		handle->_ptr = handle->_sections[(int)section];
	}
	
	b = skiprr(handle->_msg, handle->_eom, section,
		   rrnum - handle->_rrnum);
	if (b < 0)
		return (-1);
	handle->_ptr += b;
	handle->_rrnum = rrnum;

	/* Do the parse. */
	b = dn_expand(handle->_msg, handle->_eom,
		      handle->_ptr, rr->name, MAXDNAME);	// change NS_MAXDNAME to MAXDNAME
	if (b < 0)
		return (-1);
	handle->_ptr += b;
	if (handle->_ptr + INT16SZ > handle->_eom)		// change NS_INT16SZ to INT16SZ
		goto emsgsize;
	GETSHORT(rr->type, handle->_ptr);				// change NS_GET16 to GETSHORT
	if (handle->_ptr + INT16SZ > handle->_eom)
		goto emsgsize;
	GETSHORT(rr->nclass, handle->_ptr);				// change NS_GET16 to GETSHORT
	if (section == ns_s_qd) {
		rr->ttl = 0;
		rr->rdlength = 0;
		rr->rdata = NULL;
	} else {
		if (handle->_ptr + INT32SZ > handle->_eom)	// change NS_INT32SZ to INT32SZ
			goto emsgsize;
		GETLONG(rr->ttl, handle->_ptr);				// change NS_GET32 to GETLONG
		if (handle->_ptr + INT16SZ > handle->_eom)
			goto emsgsize;
		GETSHORT(rr->rdlength, handle->_ptr);		// change NS_GET16 to GETSHORT
		if (handle->_ptr + rr->rdlength > handle->_eom)
			goto emsgsize;
		rr->rdata = handle->_ptr;
		handle->_ptr += rr->rdlength;
	}
	handle->_rrnum++;

	/* All done. */
	return (0);
 enodev:
#ifndef WINNT
	errno = ENODEV;
#else
	WSASetLastError(WSAEINVAL);
	SetLastError(WSAEINVAL);
#endif
	return (-1);
 emsgsize:
#ifndef WINNT
     errno = EMSGSIZE;
#else
     WSASetLastError(WSAEMSGSIZE);
     SetLastError(WSAEMSGSIZE);
#endif
	return (-1);
}


int
ns_initparse(const u_char *msg, int msglen, ns_msg *handle) {
	const u_char *eom = msg + msglen;
	int i;

	memset(handle, 0x5e, sizeof *handle);
	handle->_msg = msg;
	handle->_eom = eom;
	if (msg + INT16SZ > eom)
		goto emsgsize;
	GETSHORT(handle->_id, msg);		// change NS_GET16 to GETSHORT
	if (msg + INT16SZ > eom)
		goto emsgsize;
	GETSHORT(handle->_flags, msg);	// change NS_GET16 to GETSHORT
	for (i = 0; i < ns_s_max; i++) {
		if (msg + INT16SZ > eom)
			goto emsgsize;
		GETSHORT(handle->_counts[i], msg);	// change NS_GET16 to GETSHORT
	}
	for (i = 0; i < ns_s_max; i++)
		if (handle->_counts[i] == 0)
			handle->_sections[i] = NULL;
		else {
			int b = skiprr(msg, eom, (ns_sect)i,
				       handle->_counts[i]);

			if (b < 0)
				return (-1);
			handle->_sections[i] = msg;
			msg += b;
		}
	if (msg != eom)
		goto emsgsize;
	handle->_sect = ns_s_max;
	handle->_rrnum = -1;
	handle->_ptr = NULL;
	return (0);
 emsgsize:
// Added #ifndef
#ifndef WINNT
     errno = EMSGSIZE;
#else
     WSASetLastError(WSAEMSGSIZE);
     SetLastError(WSAEMSGSIZE);
#endif
	return (-1);
}


// !!!!!
// caller should remember to free the pointer returned after he is done 
char **GetRData(ns_msg *handle, ns_sect section, int pflag, int *nEntries) {
	int sflag, rrnum, rdlen;
	const u_char *rdata;
	char *buf;
	ns_rr rr;      // added
	ns_type type;  // added
        char **pptr;
        int index;

	 // Print answer records.
	sflag = (_res.pfcode & pflag);
	if (_res.pfcode && !sflag)
		return NULL;
        
    // how many record returned
	rrnum = ns_msg_count(*handle, ns_s_an);   // added
#ifdef _DEBUG
	fprintf(stdout, "Number of record returned: %d\n", rrnum);
#endif
	*nEntries = rrnum;

        // malloc array of array of pointers
        pptr = (char**) malloc(sizeof(char*)*rrnum);
        //for (i=0; i<4; i++) {
        //  (char*)pptr[i] = (char *) malloc (sizeof(char)*5);
        //  memset((char*)pptr[i], '\0', 5);
        //  fprintf(stdout, "size of pptr[%d] is: %d\n", i, strlen(pptr[i])); }

        for (index=0; index<rrnum; index++) {
			if (ns_parserr(handle, section, index, &rr)) {    // added
// Added #ifndef
#ifndef WINNT
            if (errno != ENODEV)
#else
			if (WSAGetLastError() != WSAEINVAL)
#endif
              fprintf(stderr, ";; ns_parserr: %s\n",
                      strerror(errno));
            continue;
          }
          
          type = (ns_type) ns_rr_type(rr);	// added
          if (type != ns_t_txt)				// added
            continue;
          rdata = ns_rr_rdata(rr);			// added
          if (!rdata)
            continue;
          rdlen = ns_rr_rdlen(rr);			// added
          buf = (char *)malloc(rdlen);
          if (!buf)
            continue;
          
          strncpy(buf, (const char *)rdata+1, *rdata);
          buf[*rdata] = NULL;
          
          (char*)pptr[index] = buf;
          buf = NULL;
        }

        return pptr;
}


//!!! caller should free the pointer returned when done
char **GetRecordByName(const char *name, int *nEntries)
{
  union {
    HEADER hdr;
    u_char buf[PACKETSZ];
  } response;
  int responseLen = -1;
  ns_msg handle;		// added
  char **txt = NULL;

  res_init();
  responseLen = res_search(name, C_IN, T_TXT,
			  (u_char *) &response, sizeof(response));
  if (responseLen < 0) {
#ifdef _DEBUG
	fprintf(stderr, "No record(s) found...\n");
#endif
    return NULL;
  }
#ifdef _DEBUG
	fprintf(stderr, "responseLen: %i\n", responseLen);
#endif

  if (ns_initparse(response.buf, responseLen, &handle) < 0) {	// added
    fprintf(stderr, "...ns_initparse: %s\n", strerror(errno));
    return NULL;
  }

  txt = GetRData(&handle, ns_s_an, RES_PRF_ANS, nEntries);
  return txt;
}


int DeleteRecord(const char *key, char *txt, int len)
{
  ns_updrec *urec;  // missing
  u_char *data;

  res_init();
  urec = res_mkupdrec(ns_s_ud, key, C_IN, T_TXT, 0);   // missing
  urec->r_opcode = ns_uop_delete;

  if (len == -1) {
    urec->r_data = NULL;
    urec->r_size = 0;
  }
  else {
    data = (u_char*) malloc(sizeof(u_char)*(len+1));
    *data = len;
    strcpy((char*)(data+1), txt);
    urec->r_data = data;
    urec->r_size = len+1;
  }
  if (res_update(urec) < 0) {   // missing
    perror("Failed update in DeleteRecord !!\n");
    return -1;
  }

  free(urec->r_dname);
//  if (urec->r_data)
//    free(urec->r_data);
  free(urec);
  return 0;
}


int AddRecord(char *name, char *txt, int len)
{
	ns_updrec *urec;
	u_char *data;

	res_init();
	urec = res_mkupdrec(ns_s_ud, name, C_IN, T_TXT, 0);
	urec->r_opcode = ns_uop_add;

	data = (u_char*) malloc(sizeof(u_char)*(len+1));
	*data = len;
	strcpy((char *)(data+1), txt);
	urec->r_data = data;
	urec->r_size = len+1;

	if (res_update(urec) < 0) {
		perror("Failed update in AddRecord !!\n");
		return -1;
	}

	free(urec->r_dname);
	//  free(urec->r_data);
	free(urec);
	return 0;
}

/*
 * Class:     cnrg_itx_ds_DSComm
 * Method:    addRecord
 * Signature: ([B[B)V
 */
JNIEXPORT void JNICALL Java_cnrg_itx_ds_DSComm_addRecord(JNIEnv *jenv, jobject jobj, jbyteArray jentry, jbyteArray jrecord) {

	jboolean iscopy,isrec;
	jsize lenentry = jenv->GetArrayLength(jentry);
	jsize lenrecord = jenv->GetArrayLength(jrecord);
	jbyte *bodyentry = jenv->GetByteArrayElements(jentry, &iscopy);
	jbyte *bodyrecord = jenv->GetByteArrayElements(jrecord, &isrec);

#ifdef _DEBUG
	fprintf(stderr, "DEBUG: addRecord input: %s, %s\n", (char *)bodyentry, (char *)bodyrecord);
#endif
	
	AddRecord((char *)bodyentry, (char *)bodyrecord, strlen((char *)bodyrecord));

	/* free up memory */
	if (iscopy == JNI_TRUE)
		jenv->ReleaseByteArrayElements(jentry, bodyentry, 0);
	if (isrec == JNI_TRUE)
		jenv->ReleaseByteArrayElements(jrecord, bodyrecord, 0);
}

/*
 * Class:     cnrg_itx_ds_DSComm
 * Method:    deleteRecord
 * Signature: ([B[B)V
 */
JNIEXPORT void JNICALL Java_cnrg_itx_ds_DSComm_deleteRecord(JNIEnv *jenv, jobject jobj, jbyteArray jentry, jbyteArray jrecord) {

	jsize lenentry, lenrecord;
	jbyte *bodyentry, *bodyrecord;
	jboolean iscopy,isrec;


	lenentry = jenv->GetArrayLength(jentry);
	bodyentry = jenv->GetByteArrayElements(jentry, &iscopy);
	if (jrecord)
	{
		lenrecord = jenv->GetArrayLength(jrecord);
		bodyrecord = jenv->GetByteArrayElements(jrecord, &isrec);
	
	
	
#ifdef _DEBUG
		fprintf(stderr, "DEBUG: deleteRecord input: %s, %s\n", (char *)bodyentry, (char *)bodyrecord);
#endif
		DeleteRecord((char *)bodyentry, (char *)bodyrecord, strlen((char *)bodyrecord));
	}else{

#ifdef _DEBUG
		fprintf(stderr, "DEBUG: deleteRecord input: %s, null \n", (char *)bodyentry);
#endif
		DeleteRecord((char *)bodyentry, NULL, -1);
	}

	/*
	if ((char *)bodyrecord)
		DeleteRecord((char *)bodyentry, (char *)bodyrecord, strlen((char *)bodyrecord));
	else
		DeleteRecord((char *)bodyentry, NULL, -1);*/

	/* free up memory */
	if (iscopy == JNI_TRUE)
		jenv->ReleaseByteArrayElements(jentry, bodyentry, 0);
	if (isrec == JNI_TRUE)
		jenv->ReleaseByteArrayElements(jrecord, bodyrecord, 0);
}

/*
 * Class:     cnrg_itx_ds_DSComm
 * Method:    getRecord
 * Signature: ([B)Lcnrg/itx/ds/ArrayRecords;
 */
JNIEXPORT jobject JNICALL Java_cnrg_itx_ds_DSComm_getRecord(JNIEnv *jenv, jobject jobj, jbyteArray jentry) {

	jboolean iscopy;
	int index=0;
	int nSize=0;
	char **pArr;
	jsize lenentry = jenv->GetArrayLength(jentry);
	jbyte *bodyentry = jenv->GetByteArrayElements(jentry, &iscopy);

	/* create ArrayRecords object 
     Use "javap -s -p cnrg/itx/ds/ArrayRecords" command to get all methods signiture */
	jclass jstr = jenv->FindClass("cnrg/itx/ds/ArrayRecords");
	jmethodID jmIDinit = jenv->GetMethodID(jstr, "<init>", "()V");
	jmethodID jmIDadd = jenv->GetMethodID(jstr, "add", "(Ljava/lang/String;)V");
	jobject jarr = jenv->NewObject(jstr, jmIDinit);

#ifdef _DEBUG
	fprintf(stderr, "DEBUG: getRecord input: %s\n", (char *)bodyentry);
#endif
	pArr = GetRecordByName((char *)bodyentry, &nSize);

	/* set jarr elements */
	for (index=0; index<nSize; index++) {
		jstring jnewStr = jenv->NewStringUTF((char*)(pArr[index]));
		free(pArr[index]);

		jenv->CallVoidMethod(jarr, jmIDadd, jnewStr);
	}
  
	/* free up memory */
	if (iscopy == JNI_TRUE)
		jenv->ReleaseByteArrayElements(jentry, bodyentry, 0);
	free (pArr);

	return jarr;
}

/*
 * Class:     cnrg_itx_ds_DSComm
 * Method:    setConfigPath
 * Signature: ([B)V
 */
JNIEXPORT void JNICALL Java_cnrg_itx_ds_DSComm_setConfigPath(JNIEnv *jenv, jobject jobj, jbyteArray jpath) {

	jboolean iscopy;
	jsize len = jenv->GetArrayLength(jpath);
	jbyte *body = jenv->GetByteArrayElements(jpath, &iscopy);

#ifdef _DEBUG
	fprintf(stderr, "DEBUG: setConfigPath input: %s\n", (char *)body);
#endif

	init_res_paths_by_filename((char *)body);

	/* free up memory */
	if (iscopy == JNI_TRUE)
		jenv->ReleaseByteArrayElements(jpath, body, 0);
}
